USE WideWorldImporters; -- Best to avoid querying or creating objects while in master.  This may be any database of your choosing.
SET NOCOUNT ON;
GO
/*	6/30/2023 Edward Pollack
	This utility stored procedure can be used to run T-SQL across any subset of databases.  It may be customized by adjusting
	the filters that check sys.databases or database objects.

	Note that all parameters are combined with AND operators.  Therefore, it is possible to apply a variety of different
	filters to ensure that the set of databases to execute against is ideal.

	Note that while these scripts do handle names with spaces in the database name, they have not been tested for every
	name possible. Names with embedded brackets will simply be skipped and a message will be sent to warn you of this action.
*/
GO
CREATE OR ALTER PROCEDURE dbo.RunQueryAcrossDatabases
	@SQLCommand VARCHAR(MAX), -- This is the SQL to execute across any/all databases on this server.
	@SystemDatabases BIT = 1, -- Should SQL be run against system databases?  (1 = yes, 0 = no)
	@DatabaseNameLike VARCHAR(100) = NULL, -- If set, then database names will be checked against a LIKE fuzzy search of this string.
	@DatabaseNameNotLike VARCHAR(100) = NULL, -- If set, then database names will be checked against a NOT LIKE fuzzy search of this string.
	@DatabaseNameEquals VARCHAR(100) = NULL, -- If set, then only databases with this name will be considered.
	@SchemaMustContainThisObject VARCHAR(100) = NULL, -- If set, then databases will be checked for the existence of an object with this name before executing code against them.
	@SchemaCannotContainThisObject VARCHAR(100) = NULL, -- If set, then databases will be checked for the non-existence of an object with this name before executing code against them.
	@CheckOnline BIT = 1, -- When set to 1, then databases that do not have a state_desc of ONLINE will be omitted, otherwise all databases will be considered.
	@CheckMultiUser BIT = 1, -- When set to 1, then databases that do not have a user_access_desc of MULTI_USER will be omitted, otherwise all databases will be considered.
	@OnlySubstituteDatabaseNameInBrackets BIT = 0 -- When set to 1, only [?] will have the ? replaced with the database name.  This is helpful for T-SQL that includes question marks that should not be adjusted.
AS
BEGIN
	SET NOCOUNT ON;

	DECLARE @DatabaseName NVARCHAR(128); -- Stores current database name to run SQL on
	DECLARE @ValidationSQL NVARCHAR(MAX); --  Used for dynamic SQL
	DECLARE @ObjectValidationCount INT; -- Will be set later if object dependencies need to be validated.
	DECLARE @ObjectExceptionCount INT; -- Will be set later if object dependencies need to be validated.
	DECLARE @ParameterList NVARCHAR(MAX); -- For use in parameterized dynamic SQL
	-- Stores the list of databases to iterate through, after filters have been applied.
	CREATE TABLE #DatabaseNames
		(DatabaseName VARCHAR(100));

	INSERT INTO #DatabaseNames
		(DatabaseName)
	SELECT
		databases.name AS DatabaseName
	FROM sys.databases
	WHERE (@CheckOnline IS NULL OR @CheckOnline = 0 OR (@CheckOnline = 1 AND databases.state_desc = 'ONLINE'))
	AND (@CheckMultiUser IS NULL OR @CheckMultiUser = 0 OR (@CheckMultiUser = 1 AND databases.user_access_desc = 'MULTI_USER'))
	AND (@DatabaseNameLike IS NULL OR name LIKE '%' + @DatabaseNameLike + '%')
	AND (@DatabaseNameNotLike IS NULL OR name NOT LIKE '%' + @DatabaseNameNotLike + '%')
	AND (@DatabaseNameEquals IS NULL OR name = @DatabaseNameEquals)
	AND (@SystemDatabases IS NULL OR @SystemDatabases = 1 OR (@SystemDatabases = 0 AND name NOT IN ('master', 'model', 'msdb', 'tempdb')));
	
	DECLARE DBCursor CURSOR FOR SELECT DatabaseName FROM #DatabaseNames;
	OPEN DBCursor;

	FETCH NEXT FROM DBCursor INTO @DatabaseName;

	WHILE @@FETCH_STATUS = 0
	BEGIN
		IF @DatabaseName LIKE '%~[%' ESCAPE '~' OR @DatabaseName LIKE '%~]%' ESCAPE '~' 
		 BEGIN
			DECLARE @msg NVARCHAR(4000) = CONCAT('Skipping database with embedded bracket characters [ or ]. Database Name: ''',@DatabaseName,'''');
			RAISERROR (@msg,10,1);
			FETCH NEXT FROM DBCursor INTO @DatabaseName;
			CONTINUE;
		 END;

		IF @SchemaMustContainThisObject IS NOT NULL
		BEGIN
			SELECT @ValidationSQL = 'SELECT @ObjectValidationCount = COUNT(*) FROM ' + QUOTENAME(@DatabaseName) + '.[sys].[objects] WHERE [objects].[name] = ''' + @SchemaMustContainThisObject + ''';';
			SELECT @ParameterList = '@ObjectValidationCount INT OUTPUT';
			EXEC sp_executesql @ValidationSQL, @ParameterList, @ObjectValidationCount OUTPUT; -- Validate that object exists and set the count.
		END
		IF @SchemaCannotContainThisObject IS NOT NULL
		BEGIN
			SELECT @ValidationSQL = 'SELECT @ObjectExceptionCount = COUNT(*) FROM ' + QUOTENAME(@DatabaseName) + '.[sys].[objects] WHERE [objects].[name] = ''' + @SchemaCannotContainThisObject + ''';';
			SELECT @ParameterList = '@ObjectExceptionCount INT OUTPUT';
			EXEC sp_executesql @ValidationSQL, @ParameterList, @ObjectExceptionCount OUTPUT; -- Validate that object does not exist and set the count.
		END

		IF @SchemaMustContainThisObject IS NULL
		BEGIN
			SELECT @ObjectValidationCount = 1; -- If no schema validation is needed, then auto-pass this check
		END
		IF @SchemaCannotContainThisObject IS NULL
		BEGIN
			SELECT @ObjectExceptionCount = 0; -- If no schema validation is needed, then auto-pass this check
		END
		
		DECLARE @SQLCommandToExecute NVARCHAR(MAX); -- Variable to hold the SQL command that is adjusted to include the database name, where needed.
		IF EXISTS (SELECT * FROM sys.databases WHERE databases.name = @DatabaseName)
		AND @ObjectValidationCount > 0 AND @ObjectExceptionCount = 0
		BEGIN
			IF @OnlySubstituteDatabaseNameInBrackets = 1
			BEGIN
				SELECT @SQLCommandToExecute = REPLACE(@SQLCommand, '[?]', @DatabaseName); -- Replace only "[?]" with the database name
			END
			ELSE
			BEGIN
				SELECT @SQLCommandToExecute = REPLACE(@SQLCommand, '?', @DatabaseName); -- Replace "?" with the database name
			END

			EXEC sp_executesql @SQLCommandToExecute;
		END

		FETCH NEXT FROM DBCursor INTO @DatabaseName;
	END

	CLOSE DBCursor;
	DEALLOCATE DBCursor;
END
GO

-- Demo executions:

-- Get a count of user stored procedures in every database on the server.
EXEC dbo.RunQueryAcrossDatabases
	@SQLCommand = 'SELECT ''?'' AS DatabaseName, COUNT(*) FROM [?].[sys].[procedures] WHERE [procedures].[is_ms_shipped] = 0',
	@CheckOnline = 1,
	@CheckMultiUser = 1,
	@SystemDatabases = 1;
GO

-- Get a count of rows from a specific table.
EXEC dbo.RunQueryAcrossDatabases
	@SQLCommand = 'SELECT ''?'' AS DatabaseName, COUNT(*) AS PersonCount FROM [?].[Person].[Person] WHERE [Person].[EmailPromotion] <> 0',
	@DatabaseNameLike = 'AdventureWorks',
	@DatabaseNameNotLike = 'DW',
	@CheckOnline = 1,
	@CheckMultiUser = 1;
GO

-- Get the size of the largest table in each non-system/online/multi-user database.  Can adjust to return more rows using TOP N instead of TOP 1.
EXEC dbo.RunQueryAcrossDatabases
	@SQLCommand = '
		SET NOCOUNT ON;

		CREATE TABLE #StorageData
		(	TableName VARCHAR(MAX),
			RowsUsed BIGINT,
			Reserved VARCHAR(50),
			Data VARCHAR(50),
			IndexSize VARCHAR(50),
			Unused VARCHAR(50));

		INSERT INTO #StorageData
			(TableName, RowsUsed, Reserved, Data, IndexSize, Unused)
		--note the double brackets. This is needed for names with spaces in them
		--usually only necessary when publishing code in case someone names a 
		--database/column something like [To Make You Work Harder]
		EXEC [[?]].[sys].[sp_MSforeachtable] "EXEC [[?]].[sys].[sp_spaceused] ''?''";

		UPDATE #StorageData
			SET Reserved = LEFT(Reserved, LEN(Reserved) - 3),
				Data = LEFT(Data, LEN(Data) - 3),
				IndexSize = LEFT(IndexSize, LEN(IndexSize) - 3),
				Unused = LEFT(Unused, LEN(Unused) - 3);

		SELECT TOP 1
			''[?]'' AS DatabaseName,
			TableName,
			RowsUsed,
			Reserved AS data_space_reserved_kb,
			Data AS DataSpaceUsedKB,
			IndexSize AS IndexSizeKB,
			Unused AS FreeSpaceKB
		FROM #StorageData
		WHERE RowsUsed > 0
		ORDER BY CAST(Reserved AS INT) DESC;

		DROP TABLE #StorageData;
	',
	@SystemDatabases = 0,
	@CheckOnline = 1,
	@CheckMultiUser = 1,
	@OnlySubstituteDatabaseNameInBrackets = 1;
GO